/****************************************************************************/
/*     File:   ppc54xdll.h                                                  */
/*                                                                          */
/*     Purpose: This is the public header file for the C54x DSK DLL         */
/*                                                                          */
/*     Copyright (c) 1999, DNA Enterprises, Inc.                            */
/****************************************************************************/
#ifndef _PPC54X_DLL_H_
#define _PPC54X_DLL_H_

/****************************************************************************/
/* defines                                                                  */
/****************************************************************************/
/* fixup True/False defines */
#undef FALSE
#undef TRUE 
#define FALSE   ((BOOLEAN)0)
#define TRUE    ((BOOLEAN)1)

#define PPC54X_GLOBAL_EVENT_BASE_NAME   "ppc54xEvent"

#define LPT1   0
#define LPT2   1
#define LPT3   2

typedef void (PPC54X_PRINTF_FUNC)(char *);
typedef PPC54X_PRINTF_FUNC  *PPPC54X_PRINTF_FUNC;

/****************************************************************************/
/* Structures                                                               */
/****************************************************************************/
typedef struct _Register
{
    USHORT Regoff;     // Register to read/write
    USHORT DataVal;    // Data to read/write
} Register, *PRegister;

typedef struct _DskConfig {
    UCHAR   AddrReg;                   //Address register
    UCHAR   outputConfigReg;           //Output configuration register
    UCHAR   outputReg;                 //Output register
    UCHAR   soundReg;                  //Sound register
    UCHAR   inputReg;                  //Input register
    UCHAR   operationReg;              //Operation register
    UCHAR   dramBufSizeReg;            //DRAM buffer size register
    UCHAR   hostDramBufPointerReg;     //Host DRAM buffer pointer register
    UCHAR   dmaDramBufPointerReg;      //DMA DRAM buffer pointer register
    UCHAR   hostMaxBlkCntLowReg;       //Host Max Block Count Low register
    UCHAR   hostMaxBlkCntHighReg;      //Host Max Block Count high register
    UCHAR   dmaByteCntLowReg;          //DMA byte count low register
    UCHAR   dmaByteCntHighReg;         //DMA byte count high register
    UCHAR   configReg;                 //Configuration register
    UCHAR   chipRevReg;                //Chip revision register
    UCHAR   portTestReg1;              //Port test  register 1
    UCHAR   portTestReg2;              //Port test  register 2
    UCHAR   dataTxfrCtrlReg;           //Data transfer control register
} DskConfig, *PDskConfig;

/****************************************************************************/
/* Enumeration                                                              */
/****************************************************************************/
 // DSP board type definition
typedef enum {
    UNKNOWN_BOARD = 0,
    PPC_5410_BOARD = 5410,
    PPC_5402_BOARD = 5402,
} PPC54X_BOARD_TYPE, *PPPC54X_BOARD_TYPE;

//HOST-to-DSP events
typedef enum {
    _NMI_INT = 0,
    _DSP_INT,
    _DSP_INT3
} PPC54X_DSP_EVENT, *PPPC54X_DSP_EVENT;

//DSP-to-HOST events
typedef enum {
    HINT	= 0x01,
    HINT2	= 0x02
} PPC54X_HOST_EVENT, *PPPC54X_HOST_EVENT;

//DSK Semaphores
typedef enum {
    PPC54X_SEM0,
    PPC54X_SEM1
} PPC54X_SEM, *PPPC54X_SEM;

//DSK board state
typedef enum {
    PPC54X_OPEN,
    PPC54X_CLOSED,
    PPC54X_RESET,
    PPC54X_UNRESET,
    PPC54X_DSK_NOT_CONNECTED
} PPC54X_STATE, *PPPC54X_STATE;

// DSK Register select words
typedef enum {
    PPC_BUS_WRITE	          = 0x60,
    PPC_BUS_READ	          = 0x20,
    
    //PPC34C60 registers
    PPC_ADDRESS_REG           = 0x30,
    PPC_OUTPUT_CONFIG_REG     = 0x31,
    PPC_OUTPUT_REG            = 0x32,
    PPC_SOUND_REG             = 0x73,
    PPC_INPUT_REG             = 0x33,
    PPC_OPERATION_REG         = 0x34,
    PPC_DRAM_BUFSIZE_REG      = 0x35,
    PPC_HOST_DRAM_POINTER_REG = 0x36,
    PPC_DMA_DRAM_POINTER_REG  = 0x37,
    PPC_HOST_MAXCNT_LOW_REG   = 0x38,
    PPC_HOST_MAXCNT_HIGH_REG  = 0x39,
    PPC_DMA_BYTECNT_LOW_REG   = 0x3a,
    PPC_DMA_BYTECNT_HIGH_REG  = 0x33,
    PPC_CONFIG_REG            = 0x3c,
    PPC_CHIPREV_REG           = 0x3d,
    PPC_PORT_TEST_REG1        = 0x7e,
    PPC_PORT_TEST_REG2        = 0x3e,
    PPC_DATA_TXFR_CTRL_REG    = 0x3f,
    
    //HPI registers
    HPIC_REG                  = 0x00,
    HPIA_REG                  = 0x02,
    HPID_REG                  = 0x03,
    HPID_AUTO_REG             = 0x01,
    
    //TBC registers
    TBC_CONTROL0_REG          = 0x00,
    TBC_CONTROL1_REG          = 0x01,
    TBC_CONTROL2_REG          = 0x02,
    TBC_CONTROL3_REG          = 0x03,
    TBC_CONTROL4_REG          = 0x04,
    TBC_CONTROL5_REG          = 0x05,
    TBC_CONTROL6_REG          = 0x06,
    TBC_CONTROL7_REG          = 0x07,
    TBC_CONTROL8_REG          = 0x08,
    TBC_CONTROL9_REG          = 0x09,
    TBC_MAJORCMD_REG          = 0x0a,
    TBC_MINORCMD_REG          = 0x0b,
    TBC_COUNTER1_UPDATE0_REG  = 0x0c,
    TBC_COUNTER1_UPDATE1_REG  = 0x0d,
    TBC_COUNTER2_UPDATE0_REG  = 0x0e,
    TBC_COUNTER2_UPDATE1_REG  = 0x0f,
    TBC_STATUS0_REG           = 0x10,
    TBC_STATUS1_REG           = 0x11,
    TBC_STATUS2_REG           = 0x12,
    TBC_STATUS3_REG           = 0x13,
    TBC_CAPTURE0_REG          = 0x14,
    TBC_CAPTURE1_REG          = 0x15,
    TBC_READ_REG              = 0x16,
    TBC_WRITE_REG             = 0x17,

	//CPLD registers
    CPLD_CTRL_REG             = 0x00,
    CPLD_STAT1_REG            = 0x01,
    CPLD_STAT2_REG            = 0x02,
    CPLD_REV_REG              = 0x03,
    CPLD_SEM0_REG             = 0x04,
    CPLD_SEM1_REG             = 0x05

} DskRegSelect, *PDskRegSelect;

typedef enum {
    ADDR_REG_DEFAULT                   = 0x00,
    OUTPUT_CONFIG_REG_DEFAULT          = 0xff,  //select pins 42,43,44 as general purpose
                                                //outputs and pins 84, 85,86, 87 as chip 
                                                //selects.
    OUTPUT_REG_DEFAULT                 = 0x00,
    SOUND_REG_DEFAULT                  = 0x00,  // N/A
    OPERATION_REG_DEFAULT              = 0xc0,  //enable 8-bit bus access and clear any  
                                                //pending interrupt, enable interrupt latch
    DRAM_SIZE_REG_DEFAULT              = 0x0c,
    HOST_DRAM_POINTER_REG_DEFAULT      = 0x00,
    DMA_DRAM_POINTER_REG_DEFAULT       = 0x00,
    HOST_MAX_BLK_CNT_LOW_REG_DEFAULT   = 0x02,
    HOST_MAX_BLK_CNT_HIGH_REG_DEFAULT  = 0x00,
    DMA_BYTE_CNT_LOW_REG_DEFAULT       = 0x00,
    DMA_BYTE_CNT_HIGH_REG_DEFAULT      = 0x00,
    CONFIG_REG_DEFAULT                 = 0x40,  //select 16-bit DRAM data width, 
                                                //BUSCLK=SYSTEMCLK/2
												//Use MAX BLOCK COUNT to limit
												//bus accesses.
    PORT_TEST1_REG_DEFAULT             = 0x00,  //(Write only)
    DATA_TXFR_CTRL_REG_DEFAULT         = 0x01   //Reverse transfer byte mode
} PPC34C60_REG_DEFAULTS, *PPPC34C60_REG_DEFAULTS;

#undef DLL32_ENTRY
#ifdef ppc54xdll_c
    // export function, for building DLL
#define DLL32_ENTRY  __declspec(dllexport)
#else
    // import function, for using DLL
#define DLL32_ENTRY  __declspec(dllimport)
#endif

#ifdef __cplusplus

extern "C" {

#endif
    
/****************************************************************************/
/* all DLL function definitions                                             */
/****************************************************************************/

/*****************************************************************************
*  ppc54x_open()
*
*	 Description: 
*		Opens the board devices. Some functions included in the API requires an 
*		exclusive open so be sure to pass in a 1 for the exclusive flag.
*
*  Parameters:
*		board_index - corresponds to the board number for which the handle is 
*                 requested.
*		boardType - type of board to open handle to.
*
*  Return:
*		A handle to the board.
*
*******************************************************************************/
DLL32_ENTRY LPVOID ppc54x_open(USHORT port, PPC54X_BOARD_TYPE boardType);

/*****************************************************************************
*  ppc54x_close()
*
*	 Description: 
*		Closes the handle to the board. No communication with the board is possible 
*		after a call to this routine.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_close(LPVOID pDev_);

/*****************************************************************************
*  ppc54x_reset_board()
*
*	 Description: 
*		Resets the board to a known state.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_reset_board(LPVOID pDev_);

/*****************************************************************************
*  ppc54x_reset_dsp()
*
*	 Description: 
*		Holds the DSP device in reset.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_reset_dsp(LPVOID pDev_);

/*****************************************************************************
*  ppc54x_unreset_dsp()
*
*	 Description: 
*		Removes the DSP device from reset.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_unreset_dsp(LPVOID pDev_);

/*****************************************************************************
*  ppc54x_board_info()
*
*	 Description: 
*		Retrieves the board type and revision id.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		pBoardType - The board type.
*		pRevId - The revision identifier
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_board_info(LPVOID pDev_, PUSHORT pRevId, PUSHORT pBoardType); 

/*****************************************************************************
*  ppc54x_get_configuration()
*
*	 Description: 
*		Retrieves the PPC34C60 configuration information.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		pDskConfig - A pointer to a DskConfig object which contains
*		the PPC34C60 information.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_get_configuration(LPVOID pDev_, PDskConfig pDskConfig);

/*****************************************************************************
*  ppc54x_store_configuration()
*
*	 Description: 
*		Modifies the PPC34C60 configuration                                            .
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		pDskConfig - A pointer to a DskConfig object which contains
*		the PPC34C60 information.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_store_configuration(LPVOID pDev_, PDskConfig pReg);

/*****************************************************************************
*  ppc54x_hpi_read()
*
*	 Description: 
*		Reads data from the DSP memory.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		p_buffer - Pointer to the buffer to read data into.
*		p_length - Number of 16 -bit words to read.
*		src_addr - DSP memory address to read from.
*
*  Return:
*		TRUE - success.
*     FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_hpi_read(LPVOID pDev_, PULONG p_buffer, 
                                 PULONG p_len, ULONG src_addr);

/*****************************************************************************
*  ppc54x_hpi_write()
*
*	 Description: 
*		Writes data to the DSP memory.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		p_buffer - Pointer to the buffer to read data into.
*		p_length - Number of 16 -bit words to write.
*		dest_addr - DSP memory address to read from.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_hpi_write(LPVOID pDev_, PULONG p_buffer, 
                                  PULONG p_len, ULONG dest_addr);

/*****************************************************************************
*  ppc54x_generate_int()
*
*	 Description: 
*		Interrupts the DSP device.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*     intr - The type of interrupt to generate to the DSP
* 
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_generate_int(LPVOID pDev_, PPC54X_DSP_EVENT intr);

/*****************************************************************************
*  ppc54x_dll_revision()
*
*	 Description: 
*		Sets the dll revision
*
*  Parameters:
*		pRevMajor - The revison major number.
*		pRevMinor - The revison minor number.
*		pBuildNum - The build number.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_dll_revision(PULONG pRevMajor,
                                      PULONG pRevMinor,
                                      PULONG pBuildNum);

/*****************************************************************************
*  ppc54x_read_ppc_reg()
*
*	 Description: 
*		Reads a PPC34C60device (The Parallel Port controller) register.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		pReg - Pointer to the PPC34C60 register object containing the new 
*            register value.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_read_ppc_reg(LPVOID pDev_, PRegister pReg);

/*****************************************************************************
*  ppc54x_write_ppc_reg()
*
*	 Description: 
*		Modifies a PPC34C60device (The Parallel Port controller) register.
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		pReg - Pointer to the PPC34C60 register object to hold the read register 
*            value.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL ppc54x_write_ppc_reg(LPVOID pDev_, PRegister pReg);

/****************************************************************************
* ppc54x_read_cpld_reg() 
*
*	 Description: 
*  The ppc54x_read_cpld_reg function reads a value from the CPLD register
*
*  Parameters:
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		pReg - pointer to the CPLD regster that holds the read value.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_read_cpld_reg(LPVOID pDev_, PRegister pReg);

/****************************************************************************
*  ppc54x_write_cpld_reg()
*
*  The ppc54x_write_cpld_reg function writes a value to the CPLD register
*  
*		pDev_ - Board handle returned from the call to ppc54x_open.
*		pReg - pointer to a CPLD regster that holds the value to write
*
*    Returns TRUE on succes
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_write_cpld_reg(LPVOID pDev_, PRegister pReg);

/****************************************************************************/
/*                                                                          
/* ppc54x_tbc_reset() - The ppc54x_tbc_reset function pulses the TBC        
/*                       reset to reset the Test Bus Controller.            
/*                                                                          
/*		pDev_ - Board handle returned from the call to ppc54x_open.
/*
/*   returns FALSE if opreation fails                                      
/*            TRUE if successful                                            
/*                                                                          
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_tbc_reset(LPVOID pDev_);

/****************************************************************************
*  ppc54x_tbc_unreset()
*
*  The ppc54x_tbc_unreset function removes the TBC from the reset state
*
*		pDev_ - Board handle returned from the call to ppc54x_open.
*
*    Returns TRUE on succes
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_tbc_unreset(LPVOID pDev_);

/****************************************************************************
* ppc54x_read_tbc_reg()
*
*	pDev_ - Board handle returned from the call to ppc54x_open.
*	pReg - pointer to a TBC regster that holds the read value.
*
*  The ppc54x_read_reg function reads a value from the TBC device 
*  register specified by the pointer pReg
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_read_tbc_reg(LPVOID pDev_, PRegister  pReg);

/****************************************************************************
* ppc54x_write_tbc_reg()
*
*  The ppc54x_write_tbc_reg function writes a value to the TBC device 
*  register specified by the pointer pReg
*
*	pDev_ - Board handle returned from the call to ppc54x_open.
*	pReg - pointer to a TBC regster that holds the value to write.
*
*    Returns TRUE on success
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_write_tbc_reg(LPVOID pDev_, PRegister  pReg);

/****************************************************************************
* ppc54x_read_hpi_reg()
*
*  The ppc54x_read_hpi_reg function reads a value from the HPI register
*  specified by the reg.
*  
*	pDev_ - Board handle returned from the call to ppc54x_open.
*	pReg - pointer to a HPI regster that holds the read value.
*
*    Returns TRUE on success
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_read_hpi_reg(LPVOID pDev_, PRegister  pReg);

/****************************************************************************
* ppc54x_write_hpi_reg()
*
*  The ppc54x_write_hpi_reg function writes a value to the HPI register
*  specified by the reg.
*  
*	pDev_ - Board handle returned from the call to ppc54x_open.
*	pReg - pointer to a HPI regster that holds the value to write.
*  
*    Returns TRUE on succes
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_write_hpi_reg(LPVOID pDev_, PRegister  pReg);

/****************************************************************************
* ppc54x_tbc_ready()
* 
*  The ppc54x_tbc_ready function returns TRUE if the TBC ready signal is 
*  asserted and FALSE if it is not.
*
*  pDev_ - Board handle returned from the call to ppc54x_open.
*
*  
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_tbc_ready(LPVOID pDev_);

/****************************************************************************/
/*                                                                          
/* ppc54x_user_semaphore_get() - This function tries to acquire the         
/*    specified user semaphore.                                             
/*   
/*  pDev_ - Board handle returned from the call to ppc54x_open.
/*  sem - Specifies the semaphore to acquire
/* 
/*    Returns TRUE if acquired                                              
/*            FALSE if semaphore is not acquired                            
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_user_semaphore_get(LPVOID pDev_, PPC54X_SEM sem);

/****************************************************************************/
/*                                                                          
/* ppc54x_user_semaphore_release() - This function releases the previously  
/*    acquired semaphore specified by sem.                                  
/*                                                                          
/*  pDev_ - Board handle returned from the call to ppc54x_open.
/*  sem - Specifies the semaphore to release
/*                                                                          
/*   returns FALSE if the semaphore has already been released.             
/*            TRUE if operation is successful                               
/*                                                                          
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_user_semaphore_release(LPVOID pDev_, PPC54X_SEM sem);

/****************************************************************************/
/*                                                                          
/* ppc54x_user_semaphore_wait() - This function waits indefinitely to       
/*    acquire the semaphore specified by sem.                               
/*                                                                          
/*  pDev_ - Board handle returned from the call to ppc54x_open.
/*  sem - Specifies the semaphore to release
/*                                                                          
/*   returns FALSE if opreation fails                                      
/*            TRUE if successful                                            
/*                                                                          
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_user_semaphore_wait(LPVOID pDev_, PPC54X_SEM sem);

/****************************************************************************/
/*                                                                          
/* ppc54x_state() - This function returns the state of the board.           
/*                                                                          
/*  pDev_ - Board handle returned from the call to ppc54x_open.
/*
/*   returns PPC54X_STATE                                                  
/*                                                                          
/****************************************************************************/
DLL32_ENTRY PPC54X_STATE ppc54x_state(LPVOID pDev_);

/****************************************************************************/
/*                                                                          
/* ppc54x_enable_interrupts() - This function enables DSP-to-HOST interrupts.           
/*                                                                          
/*  pDev_ - Board handle returned from the call to ppc54x_open.
/*
/*   returns FALSE if opreation fails                                      
/*            TRUE if successful                                            
/*                                                                          
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_enable_interrupts(LPVOID pDev_);

/****************************************************************************/
/*                                                                          
/* ppc54x_disable_interrupts() - This function disables DSP-to-HOST interrupts.           
/*                                                                          
/*  pDev_ - Board handle returned from the call to ppc54x_open.
/*
/*   returns FALSE if opreation fails                                      
/*            TRUE if successful                                            
/*                                                                          
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_disable_interrupts(LPVOID pDev_);

/****************************************************************************/
/*                                                                          */
/* ppc54x_printf_control() - This function controls the operation of the    */
/*                          printf's (error messages) provided by this DLL. */
/*                          By default, all printf's go to the stdio device.*/
/*                          However, it is possible to disable all printf's */
/*                          or redirect them to a user function.            */
/*                                                                          */
/*    enable - When it is TRUE, printf's are enabled to stdio or redirected */
/*             function.                                                    */
/*    func   - Pointer to printf redirection function (NULL is stdio)       */
/*                                                                          */
/*    Returns TRUE.                                                         */
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_printf_control(BOOL enable,PPC54X_PRINTF_FUNC func);

/****************************************************************************/
/*
/* ppc54x_check_interrupt() - This function checks for pending interrupts  
/*                                                                 
/*  pDev_ - Board handle returned from the call to ppc54x_open.
/*
/* returns bitmask of interrupt events                                     
/*
/****************************************************************************/
DLL32_ENTRY BOOL ppc54x_check_interrupt(LPVOID pDev_);

#ifdef __cplusplus
}

#endif

#endif // _PPC54X_DLL_H_
